/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

/****************************************************************************************************
 * REMINDER
 * **************************************************************************************************
 * In order to connect with multiple channels
 * Each SGx board is expected to have a unique channel number assigned via the EEPROM.
 * They should be numbered channels 1, 2, 3, 4 respectively.
 * *************************************************************************************************/

#include <QDebug>
#include <QProcess>
#include "gui_4channel.h"
#include "miscellaneous.h"
#include "ui_gui_4channel.h"
#include "serial_v3.h"

GUI_4channel::GUI_4channel(QWidget *parent) : QWidget(parent),	ui(new Ui::GUI_4channel)
{
	ui->setupUi(this);

    ui->label_version_GUI->setText("0.14.1");

	/* Get the names of the serialports, as well as all the other startup values from config.txt */
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

	/*******************************************************************
	 * Setup channels
	 *******************************************************************/
	RF_system_constructor = new RF_System_4channel();

	/* Activate Channels */
	for (int i = 0; i < RF_System::Channels->count(); i++)
	{
		//Connect EtherCAT enable
		#ifdef ETHERCAT
			connect(this, &GUI_4channel::signal_set_ethercat_enable, &RF_system_constructor->ethercatModule, &EtherCATModule::setEnabled);
		#endif

		connect(this, &GUI_4channel::signal_channelInit, RF_System::Channels->at(i), &RF_Channel::Initialize);
		connect(this, &GUI_4channel::signal_timerStart, RF_System::Channels->at(i), &RF_Channel::Timer_Readings_Start);
		connect(this, &GUI_4channel::signal_execute_error_clear, RF_System::Channels->at(i), &RF_Channel::Execute_Error_Clear);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_firmware_version_get, this, &GUI_4channel::handler_firmware_version_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_error_get, this, &GUI_4channel::handler_error_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_temperature_PA_get, this, &GUI_4channel::handler_temperature_PA_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_temperature_SG_get, this, &GUI_4channel::handler_temperature_SG_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_RF_enable_get, this, &GUI_4channel::handler_RF_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PA_power_readings, this, &GUI_4channel::handler_PA_power_readings);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_power_get, this, &GUI_4channel::handler_power_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_frequency_get, this, &GUI_4channel::handler_frequency_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_phase_get, this, &GUI_4channel::handler_phase_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_autophase_get, this, &GUI_4channel::handler_autophase_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_autophase_enable_get, this, &GUI_4channel::handler_autophase_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_demodulator_phase_get, this, &GUI_4channel::handler_demodulator_phase_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_demodulator_enable_get, this, &GUI_4channel::handler_demodulator_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_clock_source_get, this, &GUI_4channel::handler_clock_source_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_reset_detected, this, &GUI_4channel::handler_reset_detected);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_number_get, this, &GUI_4channel::handler_channel_number_get);

		connect(this, &GUI_4channel::signal_RF_enable, RF_System::Channels->at(i), &RF_Channel::Set_RF_enable);
		connect(this, &GUI_4channel::signal_setPowerWatt, RF_System::Channels->at(i), &RF_Channel::Set_Power_Watt);
		connect(this, &GUI_4channel::signal_setFrequency, RF_System::Channels->at(i), &RF_Channel::Set_Frequency);
		connect(this, &GUI_4channel::signal_setPhase, RF_System::Channels->at(i), &RF_Channel::Set_Phase);
		connect(this, &GUI_4channel::signal_setAutoPhase, RF_System::Channels->at(i), &RF_Channel::Set_AutoPhase);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_working, this, &GUI_4channel::channel_working_handler);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_initialization_complete, this, &GUI_4channel::handler_channel_initialization_complete);


		connect(this, &GUI_4channel::signal_set_error_clearing_enable, RF_System::Channels->at(i), &RF_Channel::Set_Error_Clearing_Enable);

		emit signal_channelInit(i+1);
		emit signal_timerStart(i+1, config->get_polling_rate_data());
	}

	/*******************************************************************
	 * Load the stylesheet
	 *******************************************************************/
	QFile stylesheetfile(QCoreApplication::applicationDirPath() + "/theme.css");
	QString style;
	stylesheetfile.open(QFile::ReadOnly);
	style = stylesheetfile.readAll();
	stylesheetfile.close();

	// Set stylesheet across the board
	this->setStyleSheet(style);

	// Loads regardless of the working path. Image needs to be in the same folder as the executable.
	ui->label_logo->setStyleSheet("image: url(" + QCoreApplication::applicationDirPath() + "/logo.png);\nimage-position: top;\n");

	// Set the logo page default
	ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->page_logo));

	/*******************************************************************
	 * Button Groups
	 *******************************************************************/
	buttongroup_RCM = new QButtonGroup;
	buttongroup_RCM->addButton(ui->pushButton_RCM_1);
	buttongroup_RCM->addButton(ui->pushButton_RCM_2);
	buttongroup_RCM->addButton(ui->pushButton_RCM_3);
	buttongroup_RCM->setExclusive(true);

	ethercat_buttonGroup.addButton(ui->pushButton_etherCAT_OFF_1);
	ethercat_buttonGroup.addButton(ui->pushButton_etherCAT_ON_1);
	ethercat_buttonGroup.setExclusive(true);

	/*******************************************************************
	 * Preset Buttons
	 *******************************************************************/
	ui->pushButton_RCM_1->setChecked(true);
	ui->pushButton_etherCAT_OFF_1->setChecked(true);


	/*******************************************************************
	 * numpad setup
	 *******************************************************************/
	numpad = new Numpad(5,4,ui->label_numpad_display,ui->label_numpad_unit,ui->pushButton_numpad_period, ui->pushButton_numpad_sign);
	connect(numpad, &Numpad::value_confirmed_signal, this, &GUI_4channel::numpad_value_confirmed_handler);

	/*******************************************************************
	 * UI Arrays
	 *******************************************************************/
	frame_array[0] = ui->frame_1;
	frame_array[1] = ui->frame_2;
	frame_array[2] = ui->frame_3;
	frame_array[3] = ui->frame_4;

	readings_array[0][0] = ui->label_power_forward_1;
	readings_array[0][1] = ui->label_power_reflected_1;
	readings_array[0][2] = ui->label_s11_value_1;
	readings_array[1][0] = ui->label_power_forward_2;
	readings_array[1][1] = ui->label_power_reflected_2;
	readings_array[1][2] = ui->label_s11_value_2;
	readings_array[2][0] = ui->label_power_forward_3;
	readings_array[2][1] = ui->label_power_reflected_3;
	readings_array[2][2] = ui->label_s11_value_3;
	readings_array[3][0] = ui->label_power_forward_4;
	readings_array[3][1] = ui->label_power_reflected_4;
	readings_array[3][2] = ui->label_s11_value_4;

	demodulator_phase_array[0] = ui->label_demodulator_phase_1;
	demodulator_phase_array[1] = ui->label_demodulator_phase_2;
	demodulator_phase_array[2] = ui->label_demodulator_phase_3;
	demodulator_phase_array[3] = ui->label_demodulator_phase_4;

	RF_enable_array[0] = ui->pushButton_RF_enable_1;
	RF_enable_array[1] = ui->pushButton_RF_enable_2;
	RF_enable_array[2] = ui->pushButton_RF_enable_3;
	RF_enable_array[3] = ui->pushButton_RF_enable_4;

	power_watt_array[0] = ui->pushButton_power_watt_1;
	power_watt_array[1] = ui->pushButton_power_watt_2;
	power_watt_array[2] = ui->pushButton_power_watt_3;
	power_watt_array[3] = ui->pushButton_power_watt_4;

	frequency_array[0] = ui->pushButton_frequency_1;
	frequency_array[1] = ui->pushButton_frequency_2;
	frequency_array[2] = ui->pushButton_frequency_3;
	frequency_array[3] = ui->pushButton_frequency_4;

	phase_array[0] = ui->pushButton_phase_1;
	phase_array[1] = ui->pushButton_phase_2;
	phase_array[2] = ui->pushButton_phase_3;
	phase_array[3] = ui->pushButton_phase_4;

	autophase_array[0] = ui->pushButton_autophase_1;
	autophase_array[1] = ui->pushButton_autophase_2;
	autophase_array[2] = ui->pushButton_autophase_3;
	autophase_array[3] = ui->pushButton_autophase_4;

	stackedWidget_phase_array[0] = ui->stackedWidget_phase_1;
	stackedWidget_phase_array[1] = ui->stackedWidget_phase_2;
	stackedWidget_phase_array[2] = ui->stackedWidget_phase_3;
	stackedWidget_phase_array[3] = ui->stackedWidget_phase_4;

	error_array[0] = ui->label_error_1;
	error_array[1] = ui->label_error_2;
	error_array[2] = ui->label_error_3;
	error_array[3] = ui->label_error_4;

	temperature_PA_array[0] = ui->label_temperature_1;
	temperature_PA_array[1] = ui->label_temperature_2;
	temperature_PA_array[2] = ui->label_temperature_3;
	temperature_PA_array[3] = ui->label_temperature_4;

	temperature_SG_array[0] = ui->label_temperature_SG_1;
	temperature_SG_array[1] = ui->label_temperature_SG_2;
	temperature_SG_array[2] = ui->label_temperature_SG_3;
	temperature_SG_array[3] = ui->label_temperature_SG_4;

	clock_source_array[0] = ui->label_clock_source_1;
	clock_source_array[1] = ui->label_clock_source_2;
	clock_source_array[2] = ui->label_clock_source_3;
	clock_source_array[3] = ui->label_clock_source_4;

	channel_label_array[0] = ui->label_channel_1;
	channel_label_array[1] = ui->label_channel_2;
	channel_label_array[2] = ui->label_channel_3;
	channel_label_array[3] = ui->label_channel_4;
	channel_label_array[4] = ui->label_channel_5;
	channel_label_array[5] = ui->label_channel_6;
	channel_label_array[6] = ui->label_channel_7;
	channel_label_array[7] = ui->label_channel_8;

	/* show or hide the various UI elements */

	//
	// TODO:
	// This way of hiding the UI elements is rather error-prone if the gridlayout numbering ever changes.
	// It is also a nightmare if there are any gaps in the layout due to spacers and/or invisible mistakes.
	// It should ideally have a more resilient implementation.
	//

	/* Hide unused channel frames based on channel_select value in the config file */
	for (int i = 0; i < CHANNELCOUNT; i++)
	{
		frame_array[i]->setVisible(i < config->get_channel_count());

		for (int j = 0; j < ui->gridLayout_4->columnCount(); j++)
		{
			ui->gridLayout_4->itemAtPosition(i+1,j)->widget()->setVisible(i < config->get_channel_count());

            /* Remark:
             * The code below works for disabling the temperature readouts.
             * But if both are enabled simultaneously, an ugly gap is created for some reason.
             * Also even without the ugly gap, the spacing still seems a bit awkward when just one collumn is missing.
             */

            /* Hide The PA Temp */
            // if (j == 1 || j == 2)
            // {
            //     ui->gridLayout_4->itemAtPosition(i,j)->widget()->setVisible(config->get_read_temperature() && i-1 < config->get_channel_count());
            //     ui->gridLayout_4->itemAtPosition(i+1,j)->widget()->setVisible(config->get_read_temperature() && i < config->get_channel_count());
            // }

            /* Hide The SG Temp */
            // if (j == 3 || j == 4)
            // {
            //     ui->gridLayout_4->itemAtPosition(i,j)->widget()->setVisible(config->get_read_temperature_SG() && i-1 < config->get_channel_count());
            //     ui->gridLayout_4->itemAtPosition(i+1,j)->widget()->setVisible(config->get_read_temperature_SG() && i < config->get_channel_count());
            // }
		}
    }

    /* Populate Lists for later use */
    for (int i = 0; i < config->get_channel_count(); i++)
    {
        firmware_versions.append("");
        channel_numbers.append(0);
    }


	/* Remote Command Mode */
	RCM_supported = config->get_support_RCM_mode_USB_live();
	ui->label_RCM->setEnabled(RCM_supported);
	ui->label_RCM->setVisible(RCM_supported);
	ui->pushButton_RCM_1->setVisible(RCM_supported);
	ui->pushButton_RCM_1->setEnabled(RCM_supported);
	ui->pushButton_RCM_2->setVisible(config->get_support_RCM_mode_USB_live());
	ui->pushButton_RCM_2->setEnabled(config->get_support_RCM_mode_USB_live());
	ui->pushButton_RCM_3->setVisible(false);	// Hide the TCP button (not functional (yet))
	ui->pushButton_RCM_3->setEnabled(false);	// Hide the TCP button (not functional (yet))

	//EtherCAT
	ui->label_etherCAT_mode_1->setVisible(config->get_support_ethercat_mode());
	ui->label_etherCAT_mode_1->setEnabled(config->get_support_ethercat_mode());

	ui->pushButton_etherCAT_OFF_1->setVisible(config->get_support_ethercat_mode());
	ui->pushButton_etherCAT_ON_1->setVisible(config->get_support_ethercat_mode());
	ui->pushButton_etherCAT_OFF_1->setEnabled(config->get_support_ethercat_mode());
	ui->pushButton_etherCAT_ON_1->setEnabled(config->get_support_ethercat_mode());

	/* Constructor is done now
	 * But the initialization process continues
	 * in handler_channel_initialization_complete() */

}	//End constructor

GUI_4channel::~GUI_4channel()
{
	//Safety measure; disable the RF power (done in the channels deconstructor)
	RF_system_constructor->~RF_System_4channel();

	delete ui;
}

void GUI_4channel::handler_channel_initialization_complete(int intrasys_num, bool success)
{
	if (success == true)
	{
		/* Set the start condition for RCM. */
		handler_RCM_mode();

		/* Set the start condition for EtherCAT */
		handler_ethercat_mode();

		qDebug() << "\n\nGUI: initialization complete.";
		qDebug() << "============================\n";
	}
	else if (success == false)
	{
		qDebug() << "\n\nGUI: Subsystem[" << intrasys_num << "] Initialization failed.";
		qDebug() << "============================\n";
		if (config->get_init_failed_exit() == true)
		{
			qDebug() << "Exiting GUI.\n";
			this->close();
		}
	}
}

void GUI_4channel::handler_execute_restart_program()
{
	QProcess process;
	process.startDetached(QCoreApplication::applicationDirPath() + "/" + QCoreApplication::applicationName());
	this->close();
}

void GUI_4channel::channel_working_handler(int channel_number, bool enabled)
{
	if (enabled)
	{
		if (!frame_array[channel_number-1]->isEnabled())
		{
			frame_array[channel_number-1]->setEnabled(true);
		}
	}
	else
	{
		if (frame_array[channel_number-1]->isEnabled())
		{
			frame_array[channel_number-1]->setEnabled(false);

			//At the moment this is too prone to false alarms!
//			ui->plainTextEdit->appendPlainText("CHANNEL " + QString::number(channel_number) + " was disabled.\nCalibration may be compromised!");
		}
		//Add full set of disconnects later
	}
}




/* Enable or disable automatic error clearing.
 * 0 = error clearing is fully manual.
 * 1 = error clearing is automatic except when RCM is enabled */
void GUI_4channel::configure_auto_error_clearing()
{
	bool enable = false;
	switch(config->get_error_clear_mode())
	{
		case 0:
			enable = false;
			break;
		case 1:
			/* Use OFF buttons to determine the enable state of auto-error clearing.
			 * All OFF buttons must be checked for auto-error clearing to be enabled, otherwise no auto-error clearing. */

			enable = true;
			if (RCM_supported)
			{
				enable &= ui->pushButton_RCM_1->isChecked();
			}

			if (config->get_support_ethercat_mode())
			{
				enable &= ui->pushButton_etherCAT_OFF_1->isChecked();
			}

			break;
		default:
			enable = false;
	}

	for (int i = 0; i < RF_system_constructor->channelCount(); i++)
	{
		emit signal_set_error_clearing_enable(i+1, enable);
	}
}

void GUI_4channel::on_pushButton_clear_errors_1_clicked()
{
	for (int i = 0; i < RF_system_constructor->channelCount(); i++)
	{
		emit signal_execute_error_clear(i+1);
	}
}

/**********************************************************************************************************************************************************************************
 * GUI Button Pressing
 * *******************************************************************************************************************************************************************************/
//
//TODO:
//Main Thread should never access functions of the other threads. It's just sloppy.
//
void GUI_4channel::on_pushButton_power_watt_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Power_watt();
	numpad_parameter_select(ui->pushButton_power_watt_1, &numpad_value, "Watt");
}
void GUI_4channel::on_pushButton_power_watt_2_clicked(){
	numpad_value = RF_System::Channels->at(1)->Power_watt();
	numpad_parameter_select(ui->pushButton_power_watt_2, &numpad_value, "Watt");
}
void GUI_4channel::on_pushButton_power_watt_3_clicked(){
	numpad_value = RF_System::Channels->at(2)->Power_watt();
	numpad_parameter_select(ui->pushButton_power_watt_3, &numpad_value, "Watt");
}
void GUI_4channel::on_pushButton_power_watt_4_clicked(){
	numpad_value = RF_System::Channels->at(3)->Power_watt();
	numpad_parameter_select(ui->pushButton_power_watt_4, &numpad_value, "Watt");
}

void GUI_4channel::on_pushButton_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Frequency();
	numpad_parameter_select(ui->pushButton_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_4channel::on_pushButton_frequency_2_clicked(){
	numpad_value = RF_System::Channels->at(1)->Frequency();
	numpad_parameter_select(ui->pushButton_frequency_2, &numpad_value, "MHz", 3);
}
void GUI_4channel::on_pushButton_frequency_3_clicked(){
	numpad_value = RF_System::Channels->at(2)->Frequency();
	numpad_parameter_select(ui->pushButton_frequency_3, &numpad_value, "MHz", 3);
}
void GUI_4channel::on_pushButton_frequency_4_clicked(){
	numpad_value = RF_System::Channels->at(3)->Frequency();
	numpad_parameter_select(ui->pushButton_frequency_4, &numpad_value, "MHz", 3);
}

void GUI_4channel::on_pushButton_phase_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase();
	numpad_parameter_select(ui->pushButton_phase_1, &numpad_value, "°", 0, 1 , false);
}
void GUI_4channel::on_pushButton_phase_2_clicked(){
	numpad_value = RF_System::Channels->at(1)->Phase();
	numpad_parameter_select(ui->pushButton_phase_2, &numpad_value, "°", 0, 1 , false);
}
void GUI_4channel::on_pushButton_phase_3_clicked(){
	numpad_value = RF_System::Channels->at(2)->Phase();
	numpad_parameter_select(ui->pushButton_phase_3, &numpad_value, "°", 0, 1 , false);
}
void GUI_4channel::on_pushButton_phase_4_clicked(){
	numpad_value = RF_System::Channels->at(3)->Phase();
	numpad_parameter_select(ui->pushButton_phase_4, &numpad_value, "°", 0, 1 , false);
}

void GUI_4channel::on_pushButton_autophase_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->AutoPhase();
	numpad_parameter_select(ui->pushButton_autophase_1, &numpad_value, "°", 0, 1 , false);
}
void GUI_4channel::on_pushButton_autophase_2_clicked(){
	numpad_value = RF_System::Channels->at(1)->AutoPhase();
	numpad_parameter_select(ui->pushButton_autophase_2, &numpad_value, "°", 0, 1 , false);
}
void GUI_4channel::on_pushButton_autophase_3_clicked(){
	numpad_value = RF_System::Channels->at(2)->AutoPhase();
	numpad_parameter_select(ui->pushButton_autophase_3, &numpad_value, "°", 0, 1 , false);
}
void GUI_4channel::on_pushButton_autophase_4_clicked(){
	numpad_value = RF_System::Channels->at(3)->AutoPhase();
	numpad_parameter_select(ui->pushButton_autophase_4, &numpad_value, "°", 0, 1 , false);
}

//
//TODO:
//implement a 2 state button
//

//inverted logic because apparantly the button click + state change is handled before the clicked() slot.
void GUI_4channel::on_pushButton_RF_enable_1_clicked()
{
	emit signal_RF_enable(1, ui->pushButton_RF_enable_1->isChecked());
}

void GUI_4channel::on_pushButton_RF_enable_2_clicked()
{
	emit signal_RF_enable(2, ui->pushButton_RF_enable_2->isChecked());
}

void GUI_4channel::on_pushButton_RF_enable_3_clicked()
{
	emit signal_RF_enable(3, ui->pushButton_RF_enable_3->isChecked());
}

void GUI_4channel::on_pushButton_RF_enable_4_clicked()
{
	emit signal_RF_enable(4, ui->pushButton_RF_enable_4->isChecked());
}

/**********************************************************************************************************************************************************************************
 * NUMPAD CONFIRMED HANDLER
 * *******************************************************************************************************************************************************************************/
void GUI_4channel::numpad_value_confirmed_handler(QPushButton *button, double *num)
{
	for (int i = 0; i < RF_system_constructor->channelCount(); i++)
	{
		if (button == power_watt_array[i]){
			emit signal_setPowerWatt(i+1, *num);
		}
		if (button == frequency_array[i]){
			emit signal_setFrequency(i+1, *num);
		}
		if (button == phase_array[i]){
			emit signal_setPhase(i+1, *num);
		}
		if (button == autophase_array[i]){
			emit signal_setAutoPhase(i+1, *num);
		}
	}
}

//Numpad
void GUI_4channel::on_pushButton_numpad_0_clicked(){
	numpad->press_0();
}
void GUI_4channel::on_pushButton_numpad_1_clicked(){
	numpad->press_1();
}
void GUI_4channel::on_pushButton_numpad_2_clicked(){
	numpad->press_2();
}
void GUI_4channel::on_pushButton_numpad_3_clicked(){
	numpad->press_3();
}
void GUI_4channel::on_pushButton_numpad_4_clicked(){
	numpad->press_4();
}
void GUI_4channel::on_pushButton_numpad_5_clicked(){
	numpad->press_5();
}
void GUI_4channel::on_pushButton_numpad_6_clicked(){
	numpad->press_6();
}
void GUI_4channel::on_pushButton_numpad_7_clicked(){
	numpad->press_7();
}
void GUI_4channel::on_pushButton_numpad_8_clicked(){
	numpad->press_8();
}
void GUI_4channel::on_pushButton_numpad_9_clicked(){
	numpad->press_9();
}
void GUI_4channel::on_pushButton_numpad_period_clicked(){
	numpad->press_period();
}
void GUI_4channel::on_pushButton_numpad_backspace_clicked(){
	numpad->press_delete();
}
void GUI_4channel::on_pushButton_numpad_clear_all_clicked(){
	numpad->press_delete_all();
}
void GUI_4channel::on_pushButton_numpad_arrow_left_clicked(){
	numpad->press_indicator_left();
}
void GUI_4channel::on_pushButton_numpad_arrow_right_clicked(){
	numpad->press_indicator_right();
}
void GUI_4channel::on_pushButton_numpad_plus_clicked(){
	numpad->press_plus();
}
void GUI_4channel::on_pushButton_numpad_minus_clicked(){
	numpad->press_minus();
}
void GUI_4channel::on_pushButton_numpad_ok_clicked(){
	numpad->press_confirm_input();
	ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->page_logo));
}

void GUI_4channel::numpad_parameter_select(QPushButton *button, double *parameter, QString unit, int precision, double scale,  bool period_enable, bool sign_enable)
{
	numpad->setOutputButton(button);
	numpad->load_value(parameter, unit, period_enable, sign_enable, scale);
	ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->page_numpad));
	scale_multiplier = scale;
	this->precision = precision;
}


/**********************************************************************************************************************************************************************************
 * Channel reading handlers
 * *******************************************************************************************************************************************************************************/
void GUI_4channel::handler_firmware_version_get(int intrasys_num, int channel_num, QString version)
{
    /* Display message if the channel hasn't initialized properly */
    if (version.isEmpty())
    {
        // use intrasys_num for lack of a meaningful channel_num;
        ui->plainTextEdit->appendPlainText("CH" + QString::number(intrasys_num) + ": firmware version unknown!");
    }
    else  //Check for firmware version inconsiststencies between channels
	{
        qDebug() << "CH" + QString::number(channel_num) + ": firmware version " + version;

        channel_numbers[intrasys_num - 1] = channel_num;
        firmware_versions[intrasys_num -1] = version;

        /* Cannot continue until all channels have provided their firmware version first */
        bool gotall = true;
        for (int i = 0; i < firmware_versions.count(); i++)
        {
            if (firmware_versions.at(i) == "")
            {
                gotall = false;
                break;
            }
        }

        /* After all channels have provided their firmware version, do the comparison */
        if (gotall == true)
        {
            bool difference = false;
            QString previous_item = firmware_versions.at(0);

            for (int i = 1; i < firmware_versions.count(); i++)
            {
                if (firmware_versions.at(i) != previous_item)
                {
                    difference = true;
                    break;
                }

                previous_item = firmware_versions.at(i);
            }

            /* Display warning message if channels have mismatched firmware versions */
            if (difference == true)
            {
                ui->plainTextEdit->appendPlainText("Inconsistent firmware versions.");
                for (int i = 0; i < firmware_versions.count(); i++)
                {
                    ui->plainTextEdit->appendPlainText("CH" + QString::number(channel_numbers.at(i)) + ": firmware version " + firmware_versions.at(i));
                }
            }
        }
    }
}

void GUI_4channel::handler_error_get(int intrasys_num, quint64 error, QStringList error_messages)
{
	error_array[intrasys_num - 1]->setText("0x" + QString::number(error, 16));

	//
	// TODO:
	// Missing human-legible message indication.
	//

//	QString message_string = "";
//	for (int i = 0; i < error_messages.count(); i++)
//	{
//		if (error_messages.at(i) != "")
//		{
//			message_string += error_messages.at(i) + "\n";
//		}
//	}

//	error_messages_array[channel_number].setText(message_string);

	qDebug() << channel_label_array[intrasys_num - 1]->text() + ": Error 0x" + QString::number(error,16);
	for (int i = 0; i < error_messages.count(); i++)
	{
		if (error_messages.at(i) != "")
		{
			qDebug() << error_messages.at(i);
		}
	}

	//if all errors are cleared show 'none' errors message, else show generic error indication.
	if (error == 0)
	{
		ui->label_clear_errors_1->setText(error_messages.at(0));
	}
	else
	{
		ui->label_clear_errors_1->setText("Errors\nDetected!");
	}

	qDebug() << "";
}

void GUI_4channel::handler_reset_detected(int intrasys_num, int channel_number)
{
    (void) intrasys_num;
	ui->plainTextEdit->appendPlainText("Reset was detected on channel " + QString::number(channel_number) + ".\nPhase calibration is compromised!");
}

void GUI_4channel::handler_temperature_PA_get(int intrasys_num, double temperature)
{
	temperature_PA_array[intrasys_num - 1]->setText(QString::number(temperature,'f',1));
}

void GUI_4channel::handler_temperature_SG_get(int intrasys_num, double temperature)
{
	temperature_SG_array[intrasys_num - 1]->setText(QString::number(temperature,'f',1));
}

void GUI_4channel::handler_RF_enable_get(int intrasys_num, bool RF_enabled)
{
	if (RF_enabled == true)
	{
		RF_enable_array[intrasys_num - 1]->setText("RF is ON");
	}
	else
	{
		RF_enable_array[intrasys_num - 1]->setText("RF is OFF");
	}

	RF_enable_array[intrasys_num - 1]->setChecked(RF_enabled);
}

void GUI_4channel::handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11_loss, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_ratio)
{
    (void) PA_power_fwd_dbm, (void) PA_power_rfl_dbm, (void) PA_s11_loss;
	readings_array[intrasys_num - 1][0]->setText(QString::number(PA_power_fwd_watt,'f',1));
	readings_array[intrasys_num - 1][1]->setText(QString::number(PA_power_rfl_watt,'f',1));
	readings_array[intrasys_num - 1][2]->setText(QString::number(PA_s11_ratio,'f',1));
}

void GUI_4channel::handler_power_get(int intrasys_num, double power_dbm, double power_watt)
{
    (void) power_dbm;
	power_watt_array[intrasys_num - 1]->setText(zeroChopper(QString::number(power_watt, 'f', 2)));
}

void GUI_4channel::handler_frequency_get(int intrasys_num, double frequency_mhz)
{
	frequency_array[intrasys_num - 1]->setText(zeroChopper(QString::number(frequency_mhz, 'f', 2)));
}

void GUI_4channel::handler_phase_get(int intrasys_num, double phase_degrees)
{
	phase_array[intrasys_num - 1]->setText(zeroChopper(QString::number(phase_degrees, 'f', 1 )));
}

void GUI_4channel::handler_autophase_get(int intrasys_num, double phase_degrees)
{
	autophase_array[intrasys_num - 1]->setText(zeroChopper(QString::number(phase_degrees, 'f', 1 )));
}

void GUI_4channel::handler_demodulator_phase_get(int intrasys_num, double phase_degrees)
{
	demodulator_phase_array[intrasys_num - 1]->setText(zeroChopper(QString::number(phase_degrees, 'f', 1 )));
}

void GUI_4channel::handler_demodulator_enable_get(int intrasys_num, bool enable)
{
	if (enable == false)
	{
		demodulator_phase_array[intrasys_num - 1]->setText("-");
	}
}

/* Display the correct set of parameters depending on whether autophase is or isn't enabled.
 * enable == true -> show autophase
 * enable == false -> show regular phase */

void GUI_4channel::handler_autophase_enable_get(int intrasys_num, bool enable)
{
	if (enable == true)
	{
		if (stackedWidget_phase_array[intrasys_num - 1]->currentIndex() != 1)
		{
			stackedWidget_phase_array[intrasys_num - 1]->setCurrentIndex(1);
		}
	}
	else
	{
		if (stackedWidget_phase_array[intrasys_num - 1]->currentIndex() != 0)
		{
			stackedWidget_phase_array[intrasys_num - 1]->setCurrentIndex(0);
		}
	}
}

void GUI_4channel::handler_clock_source_get(int intrasys_num, int source)
{
	QString clock_source = "";
	switch (source)
	{
	case 0:
		clock_source = "Standalone";
		break;
	case 1:
		clock_source = "Master";
		break;
	case 2:
		clock_source = "Slave";
		break;
	case 3:
		clock_source = "Slave (Inline)";
		break;
	}
	clock_source_array[intrasys_num - 1]->setText(clock_source);
}

void GUI_4channel::handler_channel_number_get(int intrasys_num, int chan_num)
{
	channel_label_array[intrasys_num - 1]->setText("Channel " + QString::number(chan_num));
	channel_label_array[intrasys_num + 3]->setText(QString::number(chan_num));
}

/**********************************************************************************************************************************************************************************
 * Remote Command Mode
 * *******************************************************************************************************************************************************************************/
void GUI_4channel::handler_RCM_mode()
{
	if (RCM_supported == true)
	{
		switch(config->get_remote_command_mode())
		{
			case 0:
				ui->pushButton_RCM_1->click();
				break;
			case 1:
				ui->pushButton_RCM_2->click();
				break;
//			case 2:
//				ui->pushButton_remote_command_TCP_1->click();
//				break;
			default:
				ui->pushButton_RCM_1->click();
				break;
		}
	}
}

void GUI_4channel::on_pushButton_RCM_1_clicked()
{
	RF_system_constructor->RCM_USB_port->close();
	configure_auto_error_clearing();
}

void GUI_4channel::on_pushButton_RCM_2_clicked()
{
	if (!RF_system_constructor->RCM_USB_port->serial->isOpen())
	{
		if (!RF_system_constructor->RCM_USB_port->open())
		{
			ui->pushButton_RCM_1->setChecked(true);
		}
	}
	configure_auto_error_clearing();
}

/**********************************************************************************************************************************************************************************
 * EtherCAT
 *********************************************************************************************************************************************************************************/

void GUI_4channel::on_pushButton_etherCAT_OFF_1_clicked()
{

	emit signal_set_ethercat_enable(false);
	configure_auto_error_clearing();
}

void GUI_4channel::on_pushButton_etherCAT_ON_1_clicked()
{
	emit signal_set_ethercat_enable(true);
	configure_auto_error_clearing();
}

void GUI_4channel::handler_ethercat_mode()
{
	if (config->get_support_ethercat_mode() == true)
	{
		if (config->get_ethercat_enabled() == true)
		{
			ui->pushButton_etherCAT_ON_1->click();
		}
		else
		{
			ui->pushButton_etherCAT_OFF_1->click();
		}
	}
}
